#! /usr/bin/env python3
import struct
import enum
import serial
import time
        
def printMessage(s):
    return ' '.join("{:02x}".format(c) for c in s)

class MessageType(enum.Enum):
    Text = 0
    Numeric = 1
    Logic = 2

def decodeMessage(s, msgType):
    payloadSize = struct.unpack_from('<H', s, 0)[0]
    
    if payloadSize < 5:       # includes the mailSize
        raise BufferError('Payload size is too small')
    
    a,b,c,d = struct.unpack_from('<4B', s, 2)
    if a != 1 or b != 0 or c != 0x81 or d != 0x9e:
        raise BufferError('Header is not correct.  Expecting 01 00 81 9e')
    
    mailSize = struct.unpack_from('<B', s, 6)[0]
    
    if payloadSize < (5 + mailSize):  # includes the valueSize
        raise BufferError('Payload size is too small')
    
    mailBytes = struct.unpack_from('<' + str(mailSize) + 's', s, 7)[0]
    mail = mailBytes.decode('ascii')[:-1]
    
    valueSize = struct.unpack_from('<H', s, 7 + mailSize)[0]
    if payloadSize < (7 + mailSize + valueSize):  # includes the valueSize
        raise BufferError('Payload size does not match the packet')

    if msgType == MessageType.Logic:
        if valueSize != 1:
            raise BufferError('Value size is not one byte required for Logic Type')
        valueBytes = struct.unpack_from('<B', s, 9 + mailSize)[0]
        value = True if valueBytes != 0 else False
    elif msgType == MessageType.Numeric:
        if valueSize != 4:
            raise BufferError('Value size is not four bytes required for Numeric Type')
        value = struct.unpack_from('<f', s, 9 + mailSize)[0]
    else:
        valueBytes = struct.unpack_from('<' + str(valueSize) + 's', s, 9 + mailSize)[0] 
        value = valueBytes.decode('ascii')[:-1] 
        
    remnant = None
    if len(s) > (payloadSize + 2):
        remnant = s[(payloadSize) + 2:]
        
    return (mail, value, remnant)

def encodeMessage(msgType, mail, value):
    mail = mail + '\x00'
    mailBytes = mail.encode('ascii') 
    mailSize = len(mailBytes)
    fmt = '<H4BB' + str(mailSize) + 'sH'
    
    if msgType == MessageType.Logic:
        valueSize = 1
        valueBytes = 1 if value is True else 0
        fmt += 'B'
    elif msgType == MessageType.Numeric:
        valueSize = 4
        valueBytes = float(value)
        fmt += 'f'
    else:
        value = value + '\x00'
        valueBytes = value.encode('ascii')
        valueSize = len(valueBytes)
        fmt += str(valueSize) + 's'
    
    payloadSize = 7 + mailSize + valueSize
    s = struct.pack(fmt, payloadSize, 0x01, 0x00, 0x81, 0x9e, mailSize, mailBytes, valueSize, valueBytes)
    return s

####################################################################################### ROS

import numpy
import rclpy
from rclpy.node import Node
import tf_transformations as trans

from geometry_msgs.msg import Twist, PoseStamped

class Robot(Node):

    def __init__(self):
        super().__init__('robot')
        
        self.declare_parameter('port','/dev/rfcomm0') # diametro de la rueda
        self.port = self.get_parameter('port').value
        
        self.EV3 = serial.Serial(self.port)

        self.declare_parameter('wheel_diameter',55.5) # diametro de la rueda
        self.wheel_diameter = self.get_parameter('wheel_diameter').value
        self.d = self.wheel_diameter/1000

        self.declare_parameter('axle_track',119) # diametro de la rueda
        self.axle_track = self.get_parameter('axle_track').value
        self.a = self.axle_track/1000

        self.pos_left  = 0
        self.pos_right = 0
        self.pos_left_old  = 0
        self.pos_right_old = 0

        self.vel_right = 0
        self.vel_left  = 0

        self.x = 0
        self.y = 0
        self.phi = 0

        self.string_right = encodeMessage(MessageType.Numeric, 'C', self.vel_right)
        self.string_left  = encodeMessage(MessageType.Numeric, 'B', self.vel_left)

        self.declare_parameter('dt',1/20)    # seconds
        self.dt = self.get_parameter('dt').value

        self.pose_pub  = self.create_publisher(PoseStamped, 'pose', 10)
        self.timer = self.create_timer(self.dt, self.timer_callback)

        self.vel_sub = self.create_subscription(
            Twist,
            'cmd_vel',
            self.vel_callback,
            10)
        self.vel_sub  # prevent unused variable warning

    def vel_callback(self, msg):

        v = msg.linear.x
        w = msg.angular.z

        self.vel_right = int(10*(2*v+w*self.a)/self.d)
        self.vel_left  = int(10*(2*v-w*self.a)/self.d)
        self.string_right = encodeMessage(MessageType.Numeric, 'C', self.vel_right)
        self.string_left  = encodeMessage(MessageType.Numeric, 'B', self.vel_left)

        #self.get_logger().info('velocity update')

    def timer_callback(self):
        
        self.EV3.write(self.string_right)
        self.EV3.write(self.string_left)

        for i in range(10):    
            n = self.EV3.inWaiting()
            if n!= 0:
                s = self.EV3.read(n)
                # print(s)
                mail,value,s = decodeMessage(s,MessageType.Numeric)
                # print(mail,value)
                self.pos_left = value if mail == 'SB' else self.pos_left
                self.pos_right = value if mail == 'SC' else self.pos_right

        Dr = self.d*numpy.pi*(self.pos_right-self.pos_right_old)/360
        Dl = self.d*numpy.pi*(self.pos_left-self.pos_left_old)/360
        Dphi = (Dr-Dl)/self.a
        Dc = (Dr+Dl)/2

        self.x += Dc*numpy.cos(self.phi+Dphi/2)
        self.y += Dc*numpy.sin(self.phi+Dphi/2)
        self.phi += Dphi
        
        #print(self.pos_left,self.pos_right,self.x,self.y,self.phi)

        self.pos_left_old  = self.pos_left
        self.pos_right_old = self.pos_right

        pose = PoseStamped()
        pose.header.stamp = self.get_clock().now().to_msg()
        pose.header.frame_id = 'world'
        pose.pose.position.x = self.x
        pose.pose.position.y = self.y
        q = trans.quaternion_from_euler(0, 0, self.phi)
        pose.pose.orientation.x = q[0]
        pose.pose.orientation.y = q[1]
        pose.pose.orientation.z = q[2]
        pose.pose.orientation.w = q[3]
        self.pose_pub.publish(pose)

def main(args=None):
    
    rclpy.init(args=args)
    robot = Robot()
    rclpy.spin(robot)
    robot.get_logger().info('Closing')
    robot.EV3.close()
    robot.destroy_node() # Destroy the node explicitly (optional - otherwise it will be done automatically when the garbage collector destroys the node object)
    rclpy.shutdown()


if __name__ == '__main__':
    main()